<?php
/*
Plugin Name: Denmark Map Try
Plugin URI: https://www.wpmapplugins.com/interactive-map-of-denmark-wordpress-plugin.html
Description: Customize each county (colors, link, etc) through the map dashboard and use the shortcode in your page.
Version: 2.7
Author: WP Map Plugins
Author URI: https://www.wpmapplugins.com/
Text Domain: dk-map-try
Domain Path: /languages
*/

declare(strict_types=1);

namespace DKMap_TRY;

use DKMap_TRY\DKMap_TRY;

if (!defined('DKMAP_TRY_VERSION')) {
    define('DKMAP_TRY_VERSION', '2.7');
}

if (!defined('DKMAP_TRY_DIR')) {
    define('DKMAP_TRY_DIR', plugin_dir_path(__FILE__));
}

if (!defined('DKMAP_TRY_URL')) {
    define('DKMAP_TRY_URL', plugin_dir_url(__FILE__));
}

(new DKMap_TRY())->init();

class DKMap_TRY {

    const PLUGIN_NAME = 'Denmark Map Try';

    private $options = null;

    public function init() {
        $this->initActions();
        $this->initShortcodes();
        $this->initOptions();
    }

    private function initActions() {
    	if( !function_exists('wp_get_current_user') ) {
            include(ABSPATH . "wp-includes/pluggable.php");
        }
        add_action( 'admin_menu', array($this, 'addOptionsPage') );
        add_action( 'admin_footer', array($this, 'addJsConfigInFooter') );
        add_action( 'wp_footer', array($this, 'addSpanTag') );
        add_action( 'admin_enqueue_scripts', array($this, 'initAdminScript') );
        add_action( 'init', array($this, 'loadTextdomain') );
    }

    private function initShortcodes() {
        add_shortcode('dk_map_try', array($this, 'DKMap_TRYShortcode'));
    }

    private function initOptions() {
        $defaultOptions = $this->getDefaultOptions();
        $this->options  = get_option('dk_map_try');

        if (current_user_can( 'manage_options' )){
            $this->updateOptions($defaultOptions);
        }

        if (!is_array($this->options)) {
            $this->options = $defaultOptions;
        }

        $this->prepareOptionsListForTpl();
    }

    public function addJsConfigInFooter() {
        echo wp_kses_post('<span id="tipdkmap-try" style="margin-top:-32px"></span>');
        include __DIR__ . "/includes/js-config.php";
    }

    public function addOptionsPage() {
        add_menu_page(
            DKMap_TRY::PLUGIN_NAME,
            DKMap_TRY::PLUGIN_NAME,
            'manage_options',
            'dk-map-try',
            array($this, 'optionsScreen'),
            DKMAP_TRY_URL . 'public/images/map-icon.png'
        );
    }

    /**
     * @return array
     */
    private function getDefaultOptions() {
        $default = array(
            'dkbrdrclr_try'    => '#6B8B9E',
            'dkshowvisns_try'  => 1,
            'dkvisns_try'      => '#666666',
            'dkvisnshover_try' => '#113e6b',
        );

        $areas = array(
            'AARHUS', 'BORNHOLM', 'COPENHAGEN', 'COPENHAGEN MUNICIPALITY', 'FREDERIKSBERG MUNICIPALITY', 'FREDERIKSBORG', 'FUNEN', 'NORTH JUTLAND'
        );

        foreach ($areas as $k => $area) {
            $default['upclr_' . ($k + 1)]  = '#E0F3FF';
            $default['ovrclr_' . ($k + 1)] = '#8FBEE8';
            $default['dwnclr_' . ($k + 1)] = '#477CB2';
            $default['url_' . ($k + 1)]    = '';
            $default['turl_' . ($k + 1)]   = '_self';
            $default['info_' . ($k + 1)]   = $area;
            $default['enbl_' . ($k + 1)]   = 1;
        }

        return $default;
    }

    private function updateOptions(array $defaultOptions) {
        if (isset($_POST['dk_map_try']) && isset($_POST['submit-clrs'])) {
            $i = 1;
            while (isset($_POST['url_' . $i])) {
                $_POST['upclr_' . $i]  = $_POST['upclr_all'];
                $_POST['ovrclr_' . $i] = $_POST['ovrclr_all'];
                $_POST['dwnclr_' . $i] = $_POST['dwnclr_all'];

                $i++;
            }

            update_option('dk_map_try',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['dk_map_try']) && isset($_POST['submit-url'])) {
            $i = 1;
            while (isset($_POST['url_' . $i])) {
                $_POST['url_' . $i]  = $_POST['url_all'];
                $_POST['turl_' . $i] = $_POST['turl_all'];

                $i++;
            }

            update_option('dk_map_try',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['dk_map_try']) && isset($_POST['submit-info'])) {
            $i = 1;
            while (isset($_POST['url_' . $i])) {
                $_POST['info_' . $i] = $_POST['info_all'];

                $i++;
            }

            update_option('dk_map_try',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['dk_map_try']) && !isset($_POST['preview_map'])) {
            update_option('dk_map_try',$_POST);

            $this->options = $_POST;
        }

        if (isset($_POST['dk_map_try']) && isset($_POST['restore_default'])) {
            update_option('dk_map_try', $defaultOptions);

            $this->options = $defaultOptions;
        }
    }

    private function prepareOptionsListForTpl() {
        $this->options['prepared_list'] = array();
        $i                              = 1;
        while (isset($this->options['url_' . $i])) {
            $this->options['prepared_list'][] = array(
                'index'  => $i,
                'info_'  => $this->options['info_' . $i],
                'url'    => $this->options['url_' . $i],
                'turl'   => $this->options['turl_' . $i],
                'upclr'  => $this->options['upclr_' . $i],
                'ovrclr' => $this->options['ovrclr_' . $i],
                'dwnclr' => $this->options['dwnclr_' . $i],
                'enbl'   => isset($this->options['enbl_' . $i]),
            );

            $i++;
        }
    }

    public function DKMap_TRYShortcode() {
        wp_enqueue_style('dk-map-try-style-frontend', DKMAP_TRY_URL . 'public/css/map-style.css', false, '1.0', 'all');
        wp_enqueue_script('dk-map-try-interact', DKMAP_TRY_URL . 'public/js/map-interact.js?t=' . time(), array('jquery'), 10, '1.0', true);

        ob_start();

        include __DIR__ . "/includes/map.php";
        include __DIR__ . "/includes/js-config.php";

        return ob_get_clean();
    }

    public function optionsScreen() {
        include __DIR__ . "/includes/admin.php";
    }

    public function initAdminScript() {
        if ( current_user_can( 'manage_options') && ( esc_attr(isset($_GET['page'])) && esc_attr($_GET['page']) == 'dk-map-try') ):
            wp_enqueue_style('wp-color-picker');
            wp_enqueue_style('thickbox');
            wp_enqueue_script('thickbox');
            wp_enqueue_script('media-upload');

            wp_enqueue_style('dk-map-try-dashboard-style', DKMAP_TRY_URL . 'public/css/dashboard-style.css', false, '1.0', 'all');
            wp_enqueue_style('dk-map-try-style', DKMAP_TRY_URL . 'public/css/map-style.css', false, '1.0', 'all');
            wp_enqueue_style('wp-tinyeditor', DKMAP_TRY_URL . 'public/css/tinyeditor.css', false, '1.0', 'all');

            wp_enqueue_script('dk-map-try-interact', DKMAP_TRY_URL . 'public/js/map-interact.js?t=' . time(), array('jquery'), 10, '1.0', true);
            wp_enqueue_script('dk-map-try-tiny.editor', DKMAP_TRY_URL . 'public/js/editor/tinymce.min.js', 10, '1.0', true);
            wp_enqueue_script('dk-map-try-script', DKMAP_TRY_URL . 'public/js/editor/scripts.js', array('wp-color-picker'), false, true);
        endif;
    }

    public function addSpanTag()
    {
        echo wp_kses_post('<span id="tipdkmap-try"></span>');
    }
    
    public function loadTextdomain() {
        load_plugin_textdomain( 'dk-map-try', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
    }
}
